<?php
/**
 * Plugin Name: InkDNA – Fingerprinted Downloads
 * Plugin URI:  https://inkdnafingerprint.com/docs.html
 * Description: Per-buyer fingerprinting of WooCommerce downloadable files (PDF/images). Integrates with Woo download flow and marks files via the InkDNA API.
 * Version:     0.1.4
 * Author:      InkDNA
 * Author URI:  https://inkdnafingerprint.com
 * License:     GPLv2 or later
 * Text Domain: inkdna-fingerprinted-downloads
 *
 * Requires at least: 6.0
 * Tested up to:      6.8
 * Requires PHP: 8.0
 * 
 * Marking-only:
 *  1) Settings → InkDNA stores a single API Key.
 *  2) Marking is performed server-side (no client scripts). Every Woo download is fingerprinted and served locally.
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

define( 'INKDNA_FD_VERSION', '0.1.4' );
define( 'INKDNA_FD_SLUG',    'inkdna-fingerprinted-downloads' );
define( 'INKDNA_FD_OPT_BASE','inkdna_fd_api_base' );
define( 'INKDNA_FD_OPT_KEY', 'inkdna_fd_api_key' );
define( 'INKDNA_FD_OPT_MODE','inkdna_fd_mode' ); // strict|soft
define( 'INKDNA_FD_CACHE_SUBDIR', 'inkdna-fingerprinted-downloads' );

/**
 * Admin: settings page (Settings → InkDNA)
 */
add_action( 'admin_menu', function () {
    add_options_page(
        'InkDNA',
        'InkDNA',
        'manage_options',
        INKDNA_FD_SLUG,
        'inkdna_fd_render_settings'
    );
} );
// --- Sanitizers (added for reviewer-safe validation) ---
function inkdna_fd_sanitize_api_base( $v ) {
    $v = esc_url_raw( trim( (string) $v ) );
    if ( $v === '' ) { return ''; }
    $p = wp_parse_url( $v );
    if ( empty( $p['host'] ) || ( isset( $p['scheme'] ) && strtolower( $p['scheme'] ) !== 'https' ) ) {
        return ''; // require https + valid host
    }
    $host = $p['host'] . ( isset( $p['port'] ) ? ':' . (int) $p['port'] : '' );
    $path = isset( $p['path'] ) ? untrailingslashit( $p['path'] ) : '';
    return 'https://' . $host . $path; // strip query/fragment, no trailing slash
}

function inkdna_fd_sanitize_api_key( $v ) {
    $v = sanitize_text_field( (string) $v );
    // Keep common token chars only (defensive, avoids spaces/newlines)
    $v = preg_replace( '/[^A-Za-z0-9_\\-\\.=]/', '', $v );
    return substr( $v, 0, 256 );
}

function inkdna_fd_sanitize_mode( $v ) {
    $v = strtolower( sanitize_text_field( (string) $v ) );
    return in_array( $v, array( 'strict', 'soft' ), true ) ? $v : 'strict';
}

// original admin_init hook continues:
add_action( 'admin_init', function () {
    register_setting( INKDNA_FD_SLUG, INKDNA_FD_OPT_BASE, array(
        'type'              => 'string',
        'sanitize_callback' => 'inkdna_fd_sanitize_api_base',
        'default'           => 'https://ashtonx24-inkdna.hf.space',
    ) );

    register_setting( INKDNA_FD_SLUG, INKDNA_FD_OPT_KEY, array(
        'type'              => 'string',
        'sanitize_callback' => 'inkdna_fd_sanitize_api_key',
        'default'           => '',
    ) );

    register_setting( INKDNA_FD_SLUG, INKDNA_FD_OPT_MODE, array(
        'type'              => 'string',
        'sanitize_callback' => 'inkdna_fd_sanitize_mode',
        'default'           => 'strict',
    ) );

    add_settings_section(
        'inkdna_fd_main',
        'InkDNA API',
        function () {
            echo '<p>' . esc_html__( 'Server-side marking is mandatory. Configure the API host and key below.', 'inkdna-fingerprinted-downloads' ) . '</p>';
        },
        INKDNA_FD_SLUG
    );

    add_settings_field(
        INKDNA_FD_OPT_BASE,
        esc_html__( 'API Base URL', 'inkdna-fingerprinted-downloads' ),
        function () {
            $val = get_option( INKDNA_FD_OPT_BASE, 'https://ashtonx24-inkdna.hf.space' );
            echo '<input type="url" class="regular-text" name="' . esc_attr( INKDNA_FD_OPT_BASE ) . '" value="' . esc_attr( $val ) . '" placeholder="https://your-inkdna-api.example.com">';
            echo '<p class="description">' . esc_html__( 'Default: https://ashtonx24-inkdna.hf.space', 'inkdna-fingerprinted-downloads' ) . '</p>';
        },
        INKDNA_FD_SLUG,
        'inkdna_fd_main'
    );

    add_settings_field(
        INKDNA_FD_OPT_KEY,
        esc_html__( 'API Key', 'inkdna-fingerprinted-downloads' ),
        function () {
            $val = get_option( INKDNA_FD_OPT_KEY, '' );
            echo '<input type="password" class="regular-text" name="' . esc_attr( INKDNA_FD_OPT_KEY ) . '" value="' . esc_attr( $val ) . '" autocomplete="new-password">';
        },
        INKDNA_FD_SLUG,
        'inkdna_fd_main'
    );

    add_settings_field(
        INKDNA_FD_OPT_MODE,
        esc_html__( 'Mode', 'inkdna-fingerprinted-downloads' ),
        function () {
            $val = get_option( INKDNA_FD_OPT_MODE, 'strict' );
            echo '<select name="' . esc_attr( INKDNA_FD_OPT_MODE ) . '">';
            echo '<option value="strict"' . selected( $val, 'strict', false ) . '>' . esc_html__( 'Strict (block on failure)', 'inkdna-fingerprinted-downloads' ) . '</option>';
            echo '<option value="soft"' . selected( $val, 'soft', false ) . '>' . esc_html__( 'Soft (fallback to original – not recommended)', 'inkdna-fingerprinted-downloads' ) . '</option>';
            echo '</select>';
        },
        INKDNA_FD_SLUG,
        'inkdna_fd_main'
    );
} );

function inkdna_fd_render_settings() {
    if ( ! current_user_can( 'manage_options' ) ) {
        return;
    }

    echo '<div class="wrap"><h1>' . esc_html__( 'InkDNA – Fingerprinted Downloads', 'inkdna-fingerprinted-downloads' ) . '</h1>';
    echo '<form method="post" action="options.php">';
    settings_fields( INKDNA_FD_SLUG );
    do_settings_sections( INKDNA_FD_SLUG );
    submit_button( esc_html__( 'Save Settings', 'inkdna-fingerprinted-downloads' ) );
    echo '</form>';

    echo '<hr/><form method="post">';
    wp_nonce_field( 'inkdna_fd_clear_cache_action', 'inkdna_fd_clear_cache_nonce' );
    submit_button( esc_html__( 'Clear Marked Cache', 'inkdna-fingerprinted-downloads' ), 'secondary', 'inkdna_fd_clear_cache', false );
    echo '</form></div>';
}

/**
 * Handle cache clear
 */
add_action( 'admin_init', function () {
    if ( ! current_user_can( 'manage_options' ) ) {
        return;
    }
    if ( ! isset( $_POST['inkdna_fd_clear_cache'] ) ) {
        return;
    }
    if ( ! isset( $_POST['inkdna_fd_clear_cache_nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['inkdna_fd_clear_cache_nonce'] ) ), 'inkdna_fd_clear_cache_action' ) ) {
        add_action( 'admin_notices', function () {
            echo '<div class="notice notice-error"><p>' . esc_html__( 'Security check failed.', 'inkdna-fingerprinted-downloads' ) . '</p></div>';
        } );
        return;
    }

    $dir = inkdna_fd_cache_dir();
    if ( $dir && is_dir( $dir ) ) {
        $files = glob( $dir . '/*.pdf' );
        if ( is_array( $files ) ) {
            foreach ( $files as $f ) {
                if ( is_string( $f ) && file_exists( $f ) ) {
                    wp_delete_file( $f );
                }
            }
        }
    }
    add_action( 'admin_notices', function () {
        echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__( 'InkDNA marked cache cleared.', 'inkdna-fingerprinted-downloads' ) . '</p></div>';
    } );
} );

/**
 * MAIN: Intercept Woo download path and return a local, marked PDF.
 * This ensures EVERY download is fingerprinted; no bypass.
 */
add_filter( 'woocommerce_download_product_filepath', function ( $file_path, $email, $order, $product, $download ) {

    $api_base = rtrim( (string) get_option( INKDNA_FD_OPT_BASE, 'https://ashtonx24-inkdna.hf.space' ), '/' );
    $api_key  = (string) get_option( INKDNA_FD_OPT_KEY, '' );
    $mode     = (string) get_option( INKDNA_FD_OPT_MODE, 'strict' );

    if ( $api_key === '' || $api_base === '' ) {
        if ( $mode === 'soft' ) {
            return $file_path;
        }
        wp_die( esc_html__( 'Download temporarily unavailable (InkDNA not configured).', 'inkdna-fingerprinted-downloads' ), '', array( 'response' => 503 ) );
    }

    // Avoid loops if already serving a marked file
    if ( is_string( $file_path ) && str_contains( (string) $file_path, '/' . INKDNA_FD_CACHE_SUBDIR . '/' ) ) {
        return $file_path;
    }

    // Determine order context (used for FID in your API)
    $order_id = ( is_object( $order ) && method_exists( $order, 'get_id' ) ) ? (int) $order->get_id() : 0;
    if ( ! $order_id && is_user_logged_in() ) {
        $order_id = (int) get_current_user_id(); // fallback identity if no order object (rare)
    }

    // Build cache key based on source + order + download id
    $is_http = is_string( $file_path ) && preg_match( '#^https?://#i', (string) $file_path );
    $source_fingerprint = $is_http
        ? (string) $file_path
        : ( ( is_string( $file_path ) && @is_file( $file_path ) ) ? (string) md5_file( $file_path ) : (string) $file_path );

    $dl_id     = ( is_object( $download ) && method_exists( $download, 'get_id' ) ) ? (string) $download->get_id() : 'dl';
    $cache_key = 'oid' . $order_id . '-' . md5( $source_fingerprint . '|' . $dl_id );
    $dest_path = trailingslashit( inkdna_fd_cache_dir() ) . $cache_key . '.pdf';

    // Reuse cached marked file if present
    if ( file_exists( $dest_path ) && filesize( $dest_path ) > 100 ) {
        return $dest_path;
    }

    // Simple lock to avoid duplicate work on rapid clicks
    $lock_key = 'inkdna_fd_lock_' . md5( $cache_key );
    if ( get_transient( $lock_key ) ) {
        sleep( 2 );
        if ( file_exists( $dest_path ) && filesize( $dest_path ) > 100 ) {
            return $dest_path;
        }
    }
    set_transient( $lock_key, 1, 60 );

    $ok   = false;
    $resp = null;

    // Try to map uploads URL → real path for Local/localhost/dev cases.
    $uploads_map_path = inkdna_fd_map_uploads_url_to_path( (string) $file_path );
    $path_for_mark    = '';

    if ( $uploads_map_path && @is_file( $uploads_map_path ) && @is_readable( $uploads_map_path ) ) {
        $path_for_mark = $uploads_map_path; // treat as local
    } elseif ( ! $is_http && is_string( $file_path ) && @is_file( $file_path ) && @is_readable( $file_path ) ) {
        $path_for_mark = (string) $file_path; // already a local path
    }

    // Writer: validate PDF header from memory; write via WP_Filesystem
    $write_pdf = function( $pdf_bytes ) use ( $dest_path, &$ok ) {
        if ( ! is_string( $pdf_bytes ) || $pdf_bytes === '' ) {
            return;
        }
        if ( strncmp( $pdf_bytes, '%PDF', 4 ) !== 0 ) {
            if ( file_exists( $dest_path ) ) {
                wp_delete_file( $dest_path );
            }
            return;
        }
        if ( $dir = inkdna_fd_cache_dir() ) {
            $fs = inkdna_fd_fs();
            if ( $fs && method_exists( $fs, 'put_contents' ) ) {
                $written = $fs->put_contents( $dest_path, $pdf_bytes, FS_CHMOD_FILE );
                if ( $written ) {
                    $ok = true;
                }
            }
        }
    };

    // --- Attempt 1: /mark (best for Local + normal uploads) ---
    if ( $path_for_mark ) {
        $boundary = wp_generate_uuid4();
        $filename = sanitize_file_name( basename( $path_for_mark ) );
        $bits     = file_get_contents( $path_for_mark );

        $body  = "--{$boundary}\r\n";
        $body .= "Content-Disposition: form-data; name=\"file\"; filename=\"{$filename}\"\r\n";
        $body .= "Content-Type: application/pdf\r\n\r\n";
        $body .= (string) $bits . "\r\n--{$boundary}--\r\n";

        $resp = wp_remote_post( $api_base . '/mark', array(
            'headers' => array(
                'X-API-Key'    => $api_key,
                'X-Order-Id'   => (string) $order_id,
                'Content-Type' => "multipart/form-data; boundary={$boundary}",
            ),
            'body'    => $body,
            'timeout' => 90,
        ) );
        if ( ! is_wp_error( $resp ) && (int) wp_remote_retrieve_response_code( $resp ) < 300 ) {
            $write_pdf( (string) wp_remote_retrieve_body( $resp ) );
        }
    }

    // --- Attempt 2 (fallback): /mark/url ---
    if ( ! $ok ) {
        $resp = wp_remote_post( $api_base . '/mark/url', array(
            'headers' => array(
                'X-API-Key'    => $api_key,
                'X-Order-Id'   => (string) $order_id,
                'Content-Type' => 'application/json',
            ),
            'body'    => wp_json_encode( array( 'url' => (string) $file_path ), JSON_UNESCAPED_SLASHES ),
            'timeout' => 90,
        ) );

        if ( ! is_wp_error( $resp ) && (int) wp_remote_retrieve_response_code( $resp ) < 300 ) {
            $write_pdf( (string) wp_remote_retrieve_body( $resp ) );
        }
    }

    // Cleanup lock
    delete_transient( $lock_key );

    // Final decision
    if ( $ok ) {
        return $dest_path; // Woo serves the local marked copy
    }
    if ( $mode === 'soft' ) {
        return $file_path; // merchant opted in to customer-first fallback
    }

    wp_die(
        esc_html__( 'We were unable to prepare your fingerprinted download. Please try again shortly, or contact the shop owner if the issue persists.', 'inkdna-fingerprinted-downloads' ),
        '',
        array( 'response' => 502 )
    );
}, 10, 5 );

/**
 * Helpers: cache directory
 */
function inkdna_fd_cache_dir() {
    $upl = wp_upload_dir();
    $dir = trailingslashit( $upl['basedir'] ) . INKDNA_FD_CACHE_SUBDIR;
    if ( ! is_dir( $dir ) ) {
        wp_mkdir_p( $dir );
    }
    return $dir;
}

/**
 * If a URL points into the site's uploads directory, map it to a local filesystem path.
 * Returns the local path string on success, or '' if not mappable.
 */
function inkdna_fd_map_uploads_url_to_path( $maybe_url ) {
    if ( ! is_string( $maybe_url ) || $maybe_url === '' ) {
        return '';
    }
    $u = wp_upload_dir();
    if ( empty( $u['baseurl'] ) || empty( $u['basedir'] ) ) {
        return '';
    }
    // Normalize both
    $baseurl = rtrim( $u['baseurl'], '/' );
    $basedir = rtrim( $u['basedir'], DIRECTORY_SEPARATOR );

    // If the URL starts with uploads base URL, translate to local path.
    if ( preg_match( '#^https?://#i', $maybe_url ) && str_starts_with( $maybe_url, $baseurl ) ) {
        $rel = ltrim( substr( $maybe_url, strlen( $baseurl ) ), '/' );
        return $basedir . DIRECTORY_SEPARATOR . str_replace( '/', DIRECTORY_SEPARATOR, $rel );
    }
    return '';
}


/**
 * Get WP_Filesystem instance.
 */
function inkdna_fd_fs() {
    global $wp_filesystem;
    if ( ! $wp_filesystem ) {
        require_once ABSPATH . 'wp-admin/includes/file.php';
        WP_Filesystem();
    }
    return $wp_filesystem;
}
